//Source file: H:\src\Alkindi\Services\Util\SparseRatingsArray.java

package Alkindi.Services.InternalData;

import Alkindi.Data.*;
import java.util.*;

/* 
$Header: SparseRatingsArray.java, 7, 4/26/01 2:36:59 PM, Schwartz, Joe$
$Log: 
 7    Alkindi Development1.6         4/26/01 2:36:59 PM   Schwartz, Joe  
      Modifed to account for movement to new packages.
 6    Alkindi Development1.5         4/26/01 1:46:07 PM   Schwartz, Joe   Moved
      to new package.
 5    Alkindi Development1.4         2/16/01 5:50:03 PM   Schwartz, Joe   Fixed
      comments.
 4    Alkindi Development1.3         2/14/01 6:20:18 PM   Schwartz, Joe  
      Improved speed a bit and protected search arrays from modification. 
 3    Alkindi Development1.2         2/13/01 6:17:09 PM   Schwartz, Joe  
      Attempting to optimize.
 2    Alkindi Development1.1         2/13/01 5:54:45 PM   Schwartz, Joe  
      Changed to account for new SparseRatingsArray class, used for base class
      of RatingSpacePoint.
 1    Alkindi Development1.0         2/12/01 4:16:48 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
 * This class represents an array of sparse evaluation data. The data is assumed to be in the form of <tt>int</tt> Product IDs and <tt>float</tt> evaluations. Internal storage is provided by an ArrayList.<p>
 * In typical use, an instance of this class will be populated with some data, and then repeatedly accessed. Most data retrieval operations are for an evaluation for a particular product.<br>
 * For this reason, the ArrayList class is aumented with array views of the data (<tt>prodData</tt> and <tt>evalData</tt>) which are used for fast searching and retrieval. <br>
 * Data retrieval is faster using the array of Produt IDs directly because we can use the <tt>Arrays.binarySearch()</tt> method. The array must be sorted in order to perform binary searches. A sort must occur before the next retrieval operation following any call to insert or modify data.
 */
public class SparseRatingsArray implements Cloneable 
{
	
	/**
	 * Flag which indicated if data has been added to the list.<br>
	 * This member is declared private in order to prevent subclasses from erroneously clearing it without going through proper channels, so to speak. Its status can be queried via the <tt>isDirty</tt> method, and it can be set to true with the <tt>setDirty</tt> method. 
	 * If one finds you need more control over the dirty flag, one should probably make their own storage class.
	 */
	private transient boolean dirty = true;
	
	/**
	 * Array representation of the evaluation data. It is in order of the rated Products' IDs.
	 */
	protected transient float[] evalData = null;
	
	/**
	 * Array view of the product data, in order of the rated Products' IDs.
	 */
	protected transient int[] prodData = null;
	
	/**
	 * Internal storage provided by an <tt>ArrayList</tt> of <tt>ProductRating</tt> objects.
	 */
	protected ArrayList internalList;
	
	/**
	 * Implementation of <tt>clone</tt> method from the <tt>Cloneable</tt> interface. Performs a shallow copy and clones the <tt>internalList</tt> ArrayList member.
	 * @roseuid 3A8ADB87031C
	 */
	public java.lang.Object clone() 
	{
		try {
			Object newPoint = super.clone();
			SparseRatingsArray newSPR = (SparseRatingsArray)newPoint;
			newSPR.internalList = (java.util.ArrayList)internalList.clone();
			newSPR.setDirty();
			return newPoint;
		}
		catch (CloneNotSupportedException cne) {
			//	This really should NOT happen, as we have declared the class to implement 
			//	the Cloneable interface. If it does, it indicatges a serious internal error in the JVM
			//	and is handled as such.
			//
			throw new InternalError();
		}
	}
	
	/**
	 * Retrieves the evaluation for the given product ID. Returns zero for products not in the internal list. Uses <tt>Arrays.binarySearch()</tt> to locate elements faster than indexOf().
	 * @param prodID The Product ID for which an evaluation is to be found.
	 * @return A <tt>float</tt> evaluaton.
	 * @roseuid 3A884518019C
	 */
	public float getEval(int prodID) 
	{
		sort();
		int idx = Arrays.binarySearch(prodData, prodID);
		//	binarySearch() returns negative values for keys NOT found in the searched array.
		//	It should never return an index greater than or equal to the array length, but 
		//	to be sure, we check both bounds.
		//
		if (idx < 0 || idx >= prodData.length)
			return 0f;
		return evalData[idx];
	}
	
	/**
	 * Returns an array of all the evaluations in the internal list. They are sorted in order of the product IDs to which they refer.
	 * @return An array of <tt>float</tt> evaluations.
	 * @roseuid 3A88554400AB
	 */
	public float[] getEvalsArray() 
	{
		sort();
		int arrayLen = evalData.length;
		float[] retEvals = new float[arrayLen];
		System.arraycopy(evalData, 0, retEvals, 0, arrayLen);
		return retEvals;
	}
	
	/**
	 * Returns an array of all the product IDs in the internal list, sorted in order of the product IDs to which they refer.
	 * @return An array of <tt>int</tt> Product IDs.
	 * @roseuid 3A8854B8038A
	 */
	public int[] getProductArray() 
	{
		sort();
		int arrayLen = prodData.length;
		int[] retProds = new int[arrayLen];
		System.arraycopy(prodData, 0, retProds, 0, arrayLen);
		return retProds ;
	}
	
	/**
	 * Returns the status of the dirty flag. 
	 * @return
	 * @roseuid 3A8AD331034B
	 */
	public boolean isDirty() 
	{
		return dirty;
	}
	
	/**
	 * Inserts an evaluation into the internal list. If the list contains an evaluation for the given product, it is replaced by the one given here.
	 * @param prodRating A ProductRating object representing the evauation.
	 * @return void
	 * @roseuid 3A8845180197
	 */
	protected void putEval(ProductRating prodEval) 
	{
		dirty = true;
		//	Check if the an eval for the product is already in our list. indexOf()
		//	will use the equals() method of the arrayed objects.
		//	This way we limit the array to one eval per product.
		//
		int foundAt = internalList.indexOf(prodEval);
		if (foundAt > -1) {
			ProductRating foundPR = (ProductRating)internalList.get(foundAt);
			foundPR.evaluation = prodEval.evaluation;
		}
		else {
			internalList.add(prodEval);
		}
	}
	
	/**
	 * Inserts an evaluation into the internal list. If the list contains an evaluation for the given product, it is replaced by the one given here.
	 * @param prodID The Product ID for the evaluation.
	 * @oaram eval The actual evaluation.
	 * @return void
	 * @roseuid 3A8845180199
	 */
	public void putEval(int prodID, float eval) 
	{
		putEval(new ProductRating(prodID, eval));
	}
	
	/**
	 * Marks the internal list as dirty, which will force a resort operation when the next data trieval operation is attempted.
	 * @return void
	 * @roseuid 3A8AC264033C
	 */
	public void setDirty() 
	{
		dirty = true;
	}
	
	/**
	 * Returns the size of the list.
	 * @return int
	 * @roseuid 3A884711005D
	 */
	public int size() 
	{
		return internalList.size();
	}
	
	/**
	 * The internal list of sparse data must be sorted to allow data retrieval operations. This routine checks the dirty flag and sorts the internal array if necessary. It also stores the sorted objects in the objAr member, which can then be used to find elements more quickly. 
	 * @return void
	 * @roseuid 3A88451801A6
	 */
	protected void sort() 
	{
		//	Only resort if the dirty flag is set.
		//
		if (!dirty)
			return;
		//	Get the array to sort
		//
		Object[] objAr = internalList.toArray();
		Arrays.sort(objAr);
		//	Reorder internal ArrayMap and make a new array of products.
		//
		//internalList.clear();
		prodData = null;
		evalData = null;
		int arSize = objAr.length;
		prodData = new int[arSize];
		evalData = new float[arSize];
		for (int idx = 0; idx < arSize; idx ++) {
			ProductRating pr = (ProductRating)objAr[idx];
			//internalList.add(pr);
			prodData[idx] = pr.productID;
			evalData[idx] = pr.evaluation;
		}
		dirty = false;
	}
	
	/**
	 * Default constructor.
	 * @roseuid 3A8845180189
	 */
	public SparseRatingsArray() 
	{
		internalList = new ArrayList();
	}
	
	/**
	 * Constructs an array with the given initial capacity.
	 * @roseuid 3A884518018A
	 */
	public SparseRatingsArray(int capacity) 
	{
		internalList = new ArrayList(capacity);
	}
	
	/**
	 */
	static class ProductRating implements Comparable 
	{
		public float evaluation = 0;
		public int productID = 0;
		
		private ProductRating() 
		{
		}
		
		/**
		 * @roseuid 3A88451801CC
		 */
		public int compareTo(Object other) 
		{
			return this.productID - ((ProductRating)other).productID;
		}
		
		/**
		 * @roseuid 3A88451801CA
		 */
		public boolean equals(Object src) 
		{
			return ((ProductRating)src).productID == this.productID;
		}
		
		/**
		 * @roseuid 3A88451801C7
		 */
		public ProductRating(int prodID, float eval) 
		{
			productID = prodID;
			evaluation = eval;
		}
	}
}
